<?php

declare(strict_types=1);

class InstagramDownloader
{
    private array $headers = [
        'User-Agent: Mozilla/5.0 (Linux; Android 11; Mobile)',
        'Accept: text/html',
        'Accept-Language: en-US,en;q=0.9'
    ];

    private string $cookieFile;

    public function __construct(string $cookieFile = '')
    {
        $this->cookieFile = $cookieFile;
    }

    public static function respond(int $code, array $data): void
    {
        http_response_code($code);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }

    private function request(string $url, bool $useCookie = false): string
    {
        $ch = curl_init($url);

        $options = [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER     => $this->headers,
            CURLOPT_ENCODING       => '',
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT        => 20
        ];

        if ($useCookie) {
            if (!$this->cookieFile || !file_exists($this->cookieFile)) {
                throw new Exception('cookie_not_found');
            }
            $options[CURLOPT_COOKIEFILE] = $this->cookieFile;
            $options[CURLOPT_COOKIEJAR]  = $this->cookieFile;
        }

        curl_setopt_array($ch, $options);
        $res = curl_exec($ch);

        if ($res === false) {
            curl_close($ch);
            throw new Exception('request_failed');
        }

        curl_close($ch);
        return $res;
    }

    private function extractJsonBlock(string $html, string $pattern): ?array
    {
        if (!preg_match($pattern, $html, $m, PREG_OFFSET_CAPTURE)) {
            return null;
        }

        $start = $m[0][1];
        $jsonStart = strpos($html, '{', $start);
        $brace = 0;
        $length = strlen($html);

        for ($i = $jsonStart; $i < $length; $i++) {
            if ($html[$i] === '{') $brace++;
            if ($html[$i] === '}') $brace--;
            if ($brace === 0) {
                $block = substr($html, $jsonStart, $i - $jsonStart + 1);
                return json_decode($block, true);
            }
        }

        return null;
    }

    public function profile(string $url): array
    {
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            return ['status' => false, 'error' => 'invalid_url'];
        }

        try {
            $html = $this->request($url);
        } catch (Throwable $e) {
            return ['status' => false, 'error' => $e->getMessage()];
        }

        if (strpos($html, '/accounts/login/') !== false) {
            return ['status' => false, 'error' => 'blocked'];
        }

        $data = $this->extractJsonBlock($html, '/"xig_user_by_igid_v2"\s*:\s*\{/');

        if (!$data) {
            return ['status' => false, 'error' => 'user_not_found'];
        }

        return [
            'status'      => true,
            'id'          => $data['id'] ?? null,
            'pk'          => $data['pk'] ?? null,
            'username'    => $data['username'] ?? null,
            'full_name'   => $data['full_name'] ?? null,
            'biography'   => $data['biography'] ?? null,
            'followers'   => $data['follower_count'] ?? 0,
            'following'   => $data['following_count'] ?? 0,
            'profile_pic' =>
                $data['hd_profile_pic_url_info']['url']
                ?? $data['profile_pic_url_hd']
                ?? $data['profile_pic_url']
                ?? null,
            'is_verified' => $data['is_verified'] ?? false,
            'is_private'  => $data['is_private'] ?? false
        ];
    }

    public function post(string $url): array
    {
        if (!preg_match('~https?://(www\.)?instagram\.com/(p|reel|tv)/~i', $url)) {
            return ['status' => false, 'error' => 'invalid_url'];
        }

        try {
            $html = $this->request($url, true);
        } catch (Throwable $e) {
            return ['status' => false, 'error' => $e->getMessage()];
        }

        if (strpos($html, '/accounts/login/') !== false) {
            return ['status' => false, 'error' => 'blocked'];
        }

        preg_match_all('/<script type="application\/json"[^>]*data-sjs[^>]*>(.*?)<\/script>/s', $html, $m);

        $raw = null;
        foreach ($m[1] as $s) {
            if (strpos($s, 'xdt_api__v1__media__shortcode__web_info') !== false) {
                $raw = html_entity_decode($s, ENT_QUOTES);
                break;
            }
        }

        if (!$raw) {
            return ['status' => false, 'error' => 'media_not_found'];
        }

        $data = json_decode($raw, true);

        if (!$data) {
            return ['status' => false, 'error' => 'decode_failed'];
        }

        $items =
            $data['require'][0][3][0]['__bbox']['require'][0][3][1]['__bbox']
            ['result']['data']['xdt_api__v1__media__shortcode__web_info']['items'] ?? null;

        $item = $items[0] ?? null;

        if (!$item) {
            return ['status' => false, 'error' => 'empty_media'];
        }

        $typeMap = [1 => 'photo', 2 => 'video', 8 => 'carousel'];

        $pickBest = function ($c) {
            usort($c, fn($a, $b) => ($b['width'] ?? 0) <=> ($a['width'] ?? 0));
            return $c[0]['url'] ?? null;
        };

        $out = [
            'status'       => true,
            'id'           => $item['pk'] ?? null,
            'shortcode'    => $item['code'] ?? null,
            'media_type'   => $item['media_type'] ?? null,
            'type'         => $typeMap[$item['media_type']] ?? 'unknown',
            'caption'      => $item['caption']['text'] ?? null,
            'likes'        => $item['like_count'] ?? null,
            'comment_count'=> $item['comment_count'] ?? null,
            'photos_hd'    => [],
            'videos_hd'    => []
        ];

        if (!empty($item['image_versions2']['candidates'])) {
            $best = $pickBest($item['image_versions2']['candidates']);
            if ($best) $out['photos_hd'][] = $best;
        }

        if (!empty($item['video_versions'][0]['url'])) {
            $out['videos_hd'][] = $item['video_versions'][0]['url'];
        }

        if (!empty($item['carousel_media'])) {
            foreach ($item['carousel_media'] as $m) {
                if (!empty($m['image_versions2']['candidates'])) {
                    $best = $pickBest($m['image_versions2']['candidates']);
                    if ($best) $out['photos_hd'][] = $best;
                }
                if (!empty($m['video_versions'][0]['url'])) {
                    $out['videos_hd'][] = $m['video_versions'][0]['url'];
                }
            }
        }

        $out['photos_hd'] = array_values(array_unique(array_filter($out['photos_hd'])));
        $out['videos_hd'] = array_values(array_unique(array_filter($out['videos_hd'])));

        return $out;
    }
}